/* tclresolver.c - Copyright (C) 2004 Pat Thoyts <patthoyts@users.sf.net>
 *
 * Non-blocking name resolution.
 *
 * $Id: tclresolver3.c,v 1.1 2004/08/26 09:47:50 pat Exp $
 */

#define WIN32_LEAN_AND_MEAN
#include <windows.h> 
#include <winsock2.h>
#include <ws2tcpip.h>

#include <tcl.h>

#if _MSC_VER >= 1000
#pragma comment(lib, "ws2_32")
#pragma comment(lib, "user32")
#endif

#define BUFSIZE 4096
#define XWM_RESOLVE WM_APP + 1

#undef TCL_STORAGE_CLASS
#define TCL_STORAGE_CLASS DLLEXPORT

static const char ASSOC_KEY[] = "tclresolver_key";

EXTERN int Tclresolver_Init(Tcl_Interp *interp);
EXTERN int Tclresolver_Unload(Tcl_Interp *interp);
EXTERN int TclGetHostByName(LPSOCKADDR_IN saddrPtr, 
                            const char *hostname, int port);

typedef struct {
    HANDLE hReader;
    DWORD  idReader;
    Tcl_ThreadId tclid;
    Tcl_Command command;
} PKGINFO, *LPPKGINFO;

typedef struct {
    HANDLE lock;
    LONG   cookie;
    char   data[BUFSIZE];
} QUERYINFO, *LPQUERYINFO;

typedef struct {
    Tcl_Event header;
    LPQUERYINFO query;
} RESOLVEREVENT, *LPRESOLVEREVENT;

static Tcl_ObjCmdProc TclresolverObjCmd;
static Tcl_ExitProc Terminate;
static int Initialize(Tcl_Interp *interp, LPPKGINFO pkgPtr);
static Tcl_Obj *LookupHostname(LPPKGINFO pkgPtr, const char *hostname);
static DWORD WINAPI Reader(LPVOID clientData);
static void QueueEvent(Tcl_ThreadId id, LPQUERYINFO pQuery);
static int  ResolverEventProc(Tcl_Event *evPtr, int flags);
static int GetAddress(char *query);
static Tcl_Obj *Win32Error(const char * szPrefix, HRESULT hr);

/* ---------------------------------------------------------------------- */

int
Tclresolver_Init(Tcl_Interp *interp)
{
    int initialized = 0;
    LPPKGINFO pkgPtr = NULL;

#ifdef USE_TCL_STUBS
    if (Tcl_InitStubs(interp, "8.1", 0) == NULL) {
        return TCL_ERROR;
    }
#endif
    
    pkgPtr = (LPPKGINFO)Tcl_Alloc(sizeof(PKGINFO));
    if (pkgPtr == NULL) {
        Tcl_SetResult(interp, "out of memory", TCL_STATIC);
        return TCL_ERROR;
    }
    ZeroMemory(pkgPtr, sizeof(PKGINFO));

    if (Initialize(interp, pkgPtr) != TCL_OK) {
        Tcl_Free((char*)pkgPtr);
        return TCL_ERROR;
    }
    pkgPtr->tclid = Tcl_GetCurrentThread();
    pkgPtr->command = Tcl_CreateObjCommand(interp, "resolve", 
                                                TclresolverObjCmd,
                                                (ClientData)pkgPtr,
                                                (Tcl_CmdDeleteProc *)NULL);
    Tcl_CreateExitHandler(Terminate, (ClientData)pkgPtr);
    Tcl_SetAssocData(interp, ASSOC_KEY, NULL, (ClientData)pkgPtr);
    return Tcl_PkgProvide(interp, "Tclresolver", "0.4.0");
}

EXTERN int
Tclresolver_SafeInit(Tcl_Interp *interp)
{
    return Tclresolver_Init(interp);
}

int 
Tclresolver_Unload(Tcl_Interp *interp)
{
    LPPKGINFO pkgPtr = NULL;
    pkgPtr = (LPPKGINFO)Tcl_GetAssocData(interp, ASSOC_KEY, NULL);
    Tcl_DeleteCommandFromToken(interp, pkgPtr->command);
    Terminate((ClientData)pkgPtr);
    Tcl_SetAssocData(interp, ASSOC_KEY, NULL, (ClientData)NULL);    
    return TCL_OK;
}

static int
TclresolverObjCmd(ClientData clientData, Tcl_Interp *interp, 
                  int objc, Tcl_Obj *CONST objv[])
{
    LPPKGINFO pkgPtr = (LPPKGINFO)clientData;
    Tcl_Obj *resObj = NULL;
    if (objc != 2) {
        Tcl_WrongNumArgs(interp, objc, objv, "hostname");
        return TCL_ERROR;
    }
    resObj = LookupHostname(pkgPtr, Tcl_GetString(objv[1]));
    Tcl_SetObjResult(interp, resObj);
    return TCL_OK;
}

/* ---------------------------------------------------------------------- */

static int
Initialize(Tcl_Interp *interp, LPPKGINFO pkgPtr) 
{
    WSADATA wsd;
    
    OutputDebugStringA("Initialize\n");
    
    /* Initialize windows sockets. */
    if (WSAStartup(MAKEWORD(2,2), &wsd) != 0) {
	Tcl_Obj *errObj = Win32Error("WSAStartup error", GetLastError());
        Tcl_SetObjResult(interp, errObj);
        return TCL_ERROR;
    }
    
    if (LOBYTE(wsd.wVersion) != 2 ||
        HIBYTE(wsd.wVersion) != 2) {
        Tcl_SetResult(interp,
	    "error: failed to load a compatible winsock version", TCL_STATIC);
	WSACleanup();
        return TCL_ERROR;
    }

    /* Create the reader thread */
    pkgPtr->hReader = CreateThread(NULL, 0, Reader, pkgPtr,
                                        0, &pkgPtr->idReader);

    return TCL_OK;
}

static void
Terminate(ClientData clientData)
{
    LPPKGINFO pkgPtr = (LPPKGINFO)clientData;
    OutputDebugStringA("Terminate\n");
    PostThreadMessage(pkgPtr->idReader, WM_QUIT, 0, 0L);
    Tcl_DeleteExitHandler(Terminate, clientData);
    WaitForSingleObject(pkgPtr->hReader, 400);
    CloseHandle(pkgPtr->hReader);
    Tcl_Free((LPBYTE)pkgPtr);
    WSACleanup();
}

static Tcl_Obj *
LookupHostname(LPPKGINFO pkgPtr, const char * hostname)
{
    Tcl_Obj *resObj = NULL;
    DWORD dwWait;
    LPQUERYINFO queryPtr;
    static LONG cookie;
    char sz[128];

    /* copy the hostname into out buffer */
    queryPtr = (LPQUERYINFO)Tcl_Alloc(sizeof(QUERYINFO));
    memset(queryPtr, 0, sizeof(QUERYINFO));
    queryPtr->lock = CreateEvent(NULL, FALSE, FALSE, NULL);
    queryPtr->cookie = InterlockedIncrement(&cookie);
    strcpy(queryPtr->data, hostname);

    sprintf(sz, "query %d begin\n", queryPtr->cookie);
    OutputDebugString(sz);
    
    /* notify the reader thread that it should begin reading */
    PostThreadMessage(pkgPtr->idReader, XWM_RESOLVE, 0, (LPARAM)queryPtr);

    /* Run a tcl event loop while we wait for the answer */
    while (1) {
        Tcl_DoOneEvent(0); /* when tk loaded */
        //Tcl_DoOneEvent(TCL_ALL_EVENTS|TCL_DONT_WAIT);
        dwWait = WaitForSingleObject(queryPtr->lock, 0);
        if (dwWait == WAIT_OBJECT_0) {
	    sprintf(sz, "query %d complete\n", queryPtr->cookie);
	    OutputDebugString(sz);
            break;
	}
    }

    resObj = Tcl_NewStringObj(queryPtr->data, -1);
    CloseHandle(queryPtr->lock);
    Tcl_Free((char *)queryPtr);
    return resObj;
}

static DWORD WINAPI
Reader(LPVOID clientData)
{
    LPPKGINFO pkgPtr = (LPPKGINFO)clientData;
    LPQUERYINFO queryPtr = NULL;
    MSG msg;
    DWORD dwErr = 0, dwRead = 0;
    
    while(GetMessage(&msg, 0, 0, 0)) {
	if (msg.hwnd == NULL) {
	    switch (msg.message) {
		case XWM_RESOLVE:
		    queryPtr = (LPQUERYINFO)msg.lParam;
		    GetAddress(queryPtr->data);
		    QueueEvent(pkgPtr->tclid, queryPtr);
		    break;
	    }
	}
	DispatchMessage(&msg);
    }

    OutputDebugStringA("Reader exiting.\n");
    return 0;
}

static void
QueueEvent(Tcl_ThreadId id, LPQUERYINFO queryPtr)
{
    LPRESOLVEREVENT evPtr;
    OutputDebugStringA("QueueEvent\n");
    evPtr = (LPRESOLVEREVENT)Tcl_Alloc(sizeof(RESOLVEREVENT));
    evPtr->header.proc = ResolverEventProc;
    evPtr->header.nextPtr = NULL;
    evPtr->query = queryPtr;
    Tcl_ThreadQueueEvent(id, (Tcl_Event *)evPtr, TCL_QUEUE_TAIL);
}

static int
ResolverEventProc(Tcl_Event *evPtr, int flags)
{
    LPRESOLVEREVENT revPtr = (LPRESOLVEREVENT)evPtr;
    OutputDebugStringA("ResolverEventProc\n");
    SetEvent(revPtr->query->lock);
    return 1;
}

static Tcl_Obj *
Win32Error(const char * szPrefix, HRESULT hr)
{
    Tcl_Obj *msgObj = NULL;
    char * lpBuffer = NULL;
    DWORD  dwLen = 0;
    
    dwLen = FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER 
                          | FORMAT_MESSAGE_FROM_SYSTEM,
                          NULL, (DWORD)hr, LANG_NEUTRAL,
                          (LPTSTR)&lpBuffer, 0, NULL);
    if (dwLen < 1) {
        dwLen = FormatMessageA(FORMAT_MESSAGE_ALLOCATE_BUFFER 
                              | FORMAT_MESSAGE_FROM_STRING
                              | FORMAT_MESSAGE_ARGUMENT_ARRAY,
                              "code 0x%1!08X!%n", 0, LANG_NEUTRAL,
                              (LPTSTR)&lpBuffer, 0, (va_list *)&hr);
    }

    msgObj = Tcl_NewStringObj(szPrefix, -1);
    if (dwLen > 0) {
	char *p = lpBuffer + dwLen - 1;        /* remove cr-lf at end */
	for ( ; p && *p && isspace(*p); p--)
	    ;
	*++p = 0;
	Tcl_AppendToObj(msgObj, ": ", 2);
	Tcl_AppendToObj(msgObj, lpBuffer, -1);
    }
    LocalFree((HLOCAL)lpBuffer);
    return msgObj;
}

static int
GetAddress(char *query)
{
    struct addrinfo hints = {0};
    struct addrinfo *res = NULL;
    char *hostname, *p, *q;
    int r = 0, nc = 0;
    
    /* trim whitespace */
    for (p = query; *p && isspace(*p); p++)
	;
    for (q = p + strlen(p) - 1; q > p && *q && isspace(*q); q--)
	;
    *++q = 0;
    hostname = p;

    memset(&hints, 0, sizeof(hints));
    hints.ai_family = PF_INET; /* use PF_UNSPEC if we want IPv6 too */
    hints.ai_flags = AI_CANONNAME;
    hints.ai_socktype = SOCK_STREAM;
    r = getaddrinfo(hostname, "", &hints, &res);
    *query = 0;

    if (r != 0) {
	Tcl_Obj *o = Win32Error("error", WSAGetLastError());
	strcpy(query, Tcl_GetString(o));
	Tcl_DecrRefCount(o);
    } else {
	struct addrinfo *resPtr = res;
	for (resPtr = res; resPtr != NULL; resPtr = resPtr->ai_next) {
	    char name[NI_MAXHOST];

	    getnameinfo(resPtr->ai_addr, resPtr->ai_addrlen, 
			name, NI_MAXHOST,
			NULL, 0,
			NI_NUMERICHOST | NI_NUMERICSERV);

	    if (resPtr != res)
		strcat(query, " ");
	    strcat(query, name);
	}
	freeaddrinfo(res);
    }
    return r;
}
    

